import requests
import json
import time
from pathlib import Path
from datetime import datetime, timedelta, timezone
from jose import jwt
from getpass import getpass
import logging
import os

API_VERS = 3
API_VERS_STR = 'api/v{}'.format(API_VERS)

# when does a token expire
# this is the buffer time, so that the token is not used anymore if it expires soon
TOKEN_EXP_BUFFER_TIME = 600


class APILogin:

    def __init__(self, server=None, username=None, password=None):

        # path to the json file storing the accounts
        script_path = Path(os.path.dirname(os.path.realpath(__file__))).parent
        data_path =  script_path / Path('data')
        if not os.path.exists(data_path):
            os.mkdir(data_path)
        self.json_path = data_path / Path('api_login.json')

        # account defined by server, user, pass
        self._server = None
        self.username = None
        self.password = None

        self._token = None
        self._token_expiration = None
        self.accounts = None
        self._accounts_load()
        self.logger = logging.getLogger()
        
        # try to use token
        if server and username:
            server = self.__server_valid(server)
            account_test = f'{username}@{server}'
        elif username:
            server = 'https://telecovid.earconnect.de'
            account_test = f'{username}@{server}'
        else:
            account_test = None
        # token found
        if account_test is not None and account_test in list(self.accounts):
            self.account = account_test
            token = self.accounts[self.account]['token']
            if token:
                x_auth_token_expiration = datetime.fromtimestamp(jwt.get_unverified_claims(token)['exp'])
                token_time = int((x_auth_token_expiration - datetime.now()).total_seconds()) - TOKEN_EXP_BUFFER_TIME
            if token and token_time > 0:
                self.logger.info((f'reuse token from account {self.account}, it expires ' +
                             str(x_auth_token_expiration) + ' in ' +
                       str(round((x_auth_token_expiration - datetime.now()).total_seconds() / 60, 1))) + ' min')
            else:
                self.login(server=server, username=username, password=password)

        # login if username defined
        elif username:
            self.login(server=server, username=username, password=password)

        module = str(self.__class__).split("'")[1].split('.')[-2]
        class_name = str(self.__class__).split("'")[1].split('.')[-1]

        self.logger.debug('this is an instance of ' + module + '.' + class_name + '()')
        self.logger.debug('call the method help() of this instance to get help')

    @property
    def token(self):

        if not self.account and self.accounts:
            self._select_account()

        elif not self.account in self.accounts:
            return False

        token = self.accounts[self.account]['token']
        if token:
            token_exp = datetime.fromtimestamp(jwt.get_unverified_claims(token)['exp'])
            token_time = int((token_exp - datetime.now()).total_seconds()) - TOKEN_EXP_BUFFER_TIME
            #self.logger.debug('reuse token, expires in ' + str(token_time) + ' sec')

        else:
            token_time = 0

        # login if token is expired with a buffer of TOKEN_EXP_BUFFER_TIME sec
        #self.logger.debug('token expires in ' + str(token_time) + ' sec')
        if not token or token_time <= 0:
            self.login()
            if self.account in self.accounts:
                token = self.accounts[self.account]['token']
            else:
                token = False

        return token

    @property
    def base_url(self):
        return self.server + '/' + API_VERS_STR

    @property
    def account(self):

        if not self.server or not self.username:
            return None

        else:
            return self.username + '@' + self.server

    @property
    def accounts_list(self):
        return list(self.accounts)

    @account.setter
    def account(self, account):
        account_spl = account.split('@')
        self.username = account_spl[0]
        self.server = account_spl[1]

    @property
    def server(self):
        return self._server

    def __server_valid(self, server):
        if not server.startswith('http'):
            if 'localhost' in server:
                server = 'http://' + server
            else:
                server = 'https://' + server
        if server.endswith('/'):
            server = server[:-1]
        return server

    @server.setter
    def server(self, server):
        #result = self.request('version', use_token=False, server=server)
        #if result:
        #    self.logger.debug(f'{server} {result}')
        self._server = self.__server_valid(server)

    def login(self, username=None, password=None, server=None):

        if server is None and self.server is None:
            # set default server
            self._server = 'https://telecovid.earconnect.de'
        elif server:
            self.server = server

        if username:
            self.username = username

        if password:
            self.password = password

        # select account from stored accounts
        if not self.account and self.accounts:
            self._select_account()

        # enter new account data
        elif not self.account and not self.accounts:
            # is this condition for assurance of existing a self.server?!
            # in case i have used api_dc.login() i can login step by step here
            if not server:
                self.server = input('server: ')
            # if not self.server: »»» this condition will set server to telecovid by default
            #     self.server = input('server: ')

            if not self.username:
                self.username = input('username: ')

        # do the server login
        # try 3 times, if login fails?! and block user?
        login_resp = self._login_server

        # update account
        if self.account in self.accounts:

            self.accounts[self.account]['last_login'] = datetime.now()
            self.accounts[self.account]['last_login_resp'] = login_resp

        # add new account of the successful login
        elif login_resp:

            account = {
                        'server': self.server,
                        'username': self.username,
                        'last_login': datetime.now(),
                        'last_login_resp': True
                        }
            self.accounts[self.account] = account

        if login_resp:
            self.accounts[self.account]['token'] = self._token
            self.accounts[self.account]['token_expiration'] = self._token_expiration

        # dump accounts to json
        self._accounts_dump()

    def reset_redis_cache(self):
        for key in r.scan_iter('dc_api_login_password_*'):
            # delete the key
            print('del redis key ' + key.decode())
            self.r.delete(key)

    def reset_token(self):

        if self.account:

            self.accounts[self.account]['token'] = None
            self.accounts[self.account]['token_expiration'] = None

        self._accounts_dump()

    def reset_tokens(self):

        for account in self.accounts:

            self.accounts[account]['token'] = None
            self.accounts[account]['token_expiration'] = None

        self._accounts_dump()

    def _select_account(self):

        select_dic = {}
        print('Available accounts:')
        for i, account in enumerate(self.accounts):
            print('\t' + str(i+1) + '\t' + account)
            select_dic[i+1] = account
        while True:
            selected = input('select account: ')
            try:
                selected_int = int(selected)
            except ValueError:
                selected_int = False
            if selected_int in select_dic:
                self.account = select_dic[selected_int]
                break
            print('Please select a number in ' + str(list(select_dic)))

    @property
    def _login_server(self):

        self.logger.info('login: ' + str(self.username) + ' @ ' + str(self.server))
        if not self.password:
            self.password = getpass()

        resp = self.request('login',
                            use_token=False,
                            headers={'username': self.username, 'password': self.password},
                            timeout=30.05
                            )

        if resp:

            x_auth_token = resp
            x_auth_token_expiration = datetime.fromtimestamp(jwt.get_unverified_claims(x_auth_token)['exp'])
            self.logger.info(('successful login, it expires ' +
                         str(x_auth_token_expiration) + ' in ' +
                   str(round((x_auth_token_expiration - datetime.now()).total_seconds() / 60, 1))) + ' min')

            self._token = x_auth_token
            self._token_expiration = x_auth_token_expiration

            return True

        else:
            self.logger.error('login failed')
            return False

    def request(self, url_path, use_token=True, headers=None,
                data=None,files=None, json_data=None, timeout=(3.05,27), log_time=False,
                session=None, attempts=1, server=None, binary_stat=False):
        '''
        performs a API request to the labserver

            parameters:
                url_path (str): the api request url
                use_token (bool): whether to use a valid token
                headers (dict): hand over the headers to be sent
                data (str, bin): data to post
                timeout (bool, tuple): the timeout of the request
                    It’s a good practice to set connect timeouts to slightly larger than a multiple of 3,
                    which is the default TCP packet retransmission window.
                log_time (bool): whether to log the request time
                session (threading instance): use threading.local() instance for the request
                attempts (int): how many attempts for the request
                server (str): the server of the request; normally it's None, self.server is used instead
                binary_stat (??): ??

            returns:
                bool: True if successful
        '''

        time_start = time.time()
        if use_token and self.server is None:
            self.logger.debug('login attempts request')
            self.login()
        if server:
            url = server + '/' + API_VERS_STR + '/' + url_path
        elif self.server is None and use_token is False:
            self.logger.error('server is None. Maybe the server is down?')
            return False
        else:
            url = self.base_url + '/' + url_path
        self.logger.debug(url)

        attempts_max = attempts
        while attempts > 0:
            attempts -= 1

            try:
                # if session use this threading instance for the request
                if session:
                    req = session
                else:
                    req = requests

                # the default
                if use_token:
                    if not self.account or not self.account in self.accounts:
                        self.login()
                    if self.token:
                        if data:
                            response = req.post(url, data=data, headers={'X-AUTH-TOKEN': self.token}, timeout=timeout)
                        elif files:
                            response = req.post(url, files=files, headers={'X-AUTH-TOKEN': self.token}, timeout=timeout)
                        elif json_data:
                            response = req.post(url, json=json_data, headers={'X-AUTH-TOKEN': self.token}, timeout=timeout)
                        else:
                            response = req.get(url, headers={'X-AUTH-TOKEN': self.token}, timeout=timeout)
                    else:
                        self.logger.error(url + ' request failed, no valid token')
                        return False
                # actually used for the login
                elif headers:
                    response = req.get(url, headers=headers, timeout=timeout)
                # for requests without auth
                else:
                    response = req.get(url, timeout=timeout)

                if binary_stat:
                    response_str = response
                else:
                    response_str = response.content.decode()


            # all exceptions from the module requests inherit from requests.exceptions.RequestException
            except requests.exceptions.RequestException as e:
                if attempts > 0:
                    self.logger.warning(url + ' failed with RequestException: ' + str(e) + ' attempts left: ' + str(attempts))
                    continue
                else:
                    self.logger.error(url + ' failed permanently with RequestException: ' + str(e))
                    return False

            if response.status_code == 200:
                if log_time:
                    req_time = round(time.time()-time_start, 1)
                    self.logger.debug('{} time elapsed {} s, attempts: {}'.format(url, req_time, attempts_max-attempts))

                if binary_stat:
                    return response_str
                else:
                    resp = json.loads(response_str)

                    if 'x_auth_token' in resp:
                        return resp['x_auth_token']
                    elif 'error' in resp:
                        self.logger.warning(resp['error'])
                        return False
                    elif 'data' in resp:
                        return resp['data']
                    else:
                        self.logger.error(url + ' server response unknown: ' + str(resp))
                        return False


            elif response.status_code == 500:
                self.logger.error(url + ' failed with http code ' +
                               str(response.status_code) + ', server error.')
                return False

            elif response.status_code == 401:
                try:
                    error = json.loads(response_str)['error']
                except Exception as e:
                    self.logger.error('Unknown error: ' + str(e))
                    error = 'unknown'
                if 'x_auth_token invalid' in error:
                    self.logger.info('reset token of account: ' + str(self.account))
                    self.reset_token()
                self.logger.warning(url + ' failed with http code ' +
                               str(response.status_code) + ', access denied. Error: ' +
                               error)
                return False

            elif response.status_code == 403:
                try:
                    error = json.loads(response_str)['error']
                except Exception as e:
                    self.logger.error('Unknown error: ' + str(e))
                    error = 'unknown'
                self.logger.warning(url + ' failed with http code ' +
                               str(response.status_code) + ', access denied. Error: ' +
                               error)
                return False

            elif response.status_code == 404:
                self.logger.error(url + ' failed with http code ' +
                               str(response.status_code) + ', route not found.')
                return False

            else:
                self.logger.error(url + ' failed with http code ' +
                               str(response.status_code) + ', response ' +
                               str(response_str))
                return False

    def _accounts_load(self):

        try:
            self.accounts = json.load(open(str(self.json_path)))
        except (FileNotFoundError, json.decoder.JSONDecodeError):
            self.accounts = {}

    def _accounts_dump(self):

        json.dump(self.accounts, open(str(self.json_path), 'w'), indent=4, default=str)

